﻿#pragma execution_character_set("utf-8")
#include "script_parser.h"
#include "script_functions.h"
#include <fstream>
#include <regex>
#include <iostream>
#include <nlohmann/json.hpp>  // 添加这一行

using json = nlohmann::json;  // 添加这一行

// 删除这些函数声明，因为它们已经在 nba2kol2_keys.h 中声明
// std::wstring Utf8ToWide(const std::string& utf8Str);
// std::string WideToUtf8(const std::wstring& wstr);

// 初始化键名映射表
std::unordered_map<std::string, int> ScriptParser::keyNameMap;

// 添加初始化方法实现
void ScriptParser::InitKeyNameMap() {
    keyNameMap.clear();
    keyNameMap = {
        {"冲刺键", g_keys.sprint_key},
        {"投篮键", g_keys.shoot_key},
        {"背身键", g_keys.post_up_key},
        {"传球键", g_keys.pass_key},
        {"击地传球键", g_keys.bounce_pass_key},
        {"高吊传球键", g_keys.lob_pass_key},
        {"向上移动球员键", g_keys.move_up_key},
        {"向下移动球员键", g_keys.move_down_key},
        {"向左移动球员键", g_keys.move_left_key},
        {"向右移动球员键", g_keys.move_right_key},
        {"特殊方向上键", g_keys.special_up_key},
        {"特殊方向下键", g_keys.special_down_key},
        {"特殊方向左键", g_keys.special_left_key},
        {"特殊方向右键", g_keys.special_right_key},
        {"特殊方向左上键", g_keys.special_left_up_key},
        {"特殊方向左下键", g_keys.special_left_down_key},
        {"特殊方向右下键", g_keys.special_right_down_key},
        {"特殊方向右上键", g_keys.special_right_up_key},
        {"右手持球半转身键", g_keys.right_hand_spin_key},
        {"左手持球半转身键", g_keys.left_hand_spin_key},
        {"右手持球转身键", g_keys.right_hand_turn_key},
        {"左手持球转身键", g_keys.left_hand_turn_key}
    };
}

// 修改从文件加载脚本的函数，正确处理中文文件名
CompiledScript ScriptParser::LoadFromFile(const std::string& filePath) {
    // 在解析脚本前初始化键位映射表
    InitKeyNameMap();

    // 使用二进制模式打开文件
    std::ifstream file(filePath, std::ios::binary);
    if (!file.is_open()) {
        throw std::runtime_error("无法打开脚本文件: " + filePath);
    }

    // 检测UTF-8 BOM
    char bom[3];
    bool hasBom = false;
    if (file.read(bom, 3) && bom[0] == (char)0xEF && bom[1] == (char)0xBB && bom[2] == (char)0xBF) {
        hasBom = true;
    }
    else {
        file.seekg(0); // 如果没有BOM，重置文件指针到开头
    }

    // 读取整个文件内容
    std::string content((std::istreambuf_iterator<char>(file)), std::istreambuf_iterator<char>());
    file.close();

    // 提取文件名作为脚本名称（去掉路径和扩展名）
    std::string name = filePath;
    size_t lastSlash = name.find_last_of("/\\");
    if (lastSlash != std::string::npos) {
        name = name.substr(lastSlash + 1);
    }
    size_t lastDot = name.find_last_of(".");
    if (lastDot != std::string::npos) {
        name = name.substr(0, lastDot);
    }

    // 检查名称是否为空，如果为空则使用默认名称
    if (name.empty()) {
        name = "未命名脚本";
    }

    // 从脚本内容中提取描述信息
    std::string description;
    std::regex descRegex("#脚本功能:(.*)");
    std::smatch match;
    if (std::regex_search(content, match, descRegex) && match.size() > 1) {
        description = match[1].str();
        // 去除前后空格
        description = std::regex_replace(description, std::regex("^\\s+|\\s+$"), "");
    }

    // 如果描述为空，设置默认描述
    if (description.empty()) {
        description = "";
    }

    return ParseScript(content, name, description);
}

// 添加宽字符版本的LoadFromFile函数实现
CompiledScript ScriptParser::LoadFromFile(const std::wstring& filePath) {
    // 在解析脚本前初始化键位映射表
    InitKeyNameMap();

    // 使用宽字符版本的API打开文件
    std::ifstream file(filePath, std::ios::binary);
    if (!file.is_open()) {
        throw std::runtime_error("无法打开脚本文件");
    }

    // 检测UTF-8 BOM
    char bom[3];
    bool hasBom = false;
    if (file.read(bom, 3) && bom[0] == (char)0xEF && bom[1] == (char)0xBB && bom[2] == (char)0xBF) {
        hasBom = true;
    }
    else {
        file.seekg(0); // 如果没有BOM，重置文件指针到开头
    }

    // 读取整个文件内容
    std::string content((std::istreambuf_iterator<char>(file)), std::istreambuf_iterator<char>());
    file.close();

    // 提取文件名作为脚本名称
    std::string name = WideToUtf8(filePath);
    size_t lastSlash = name.find_last_of("/\\");
    if (lastSlash != std::string::npos) {
        name = name.substr(lastSlash + 1);
    }
    size_t lastDot = name.find_last_of(".");
    if (lastDot != std::string::npos) {
        name = name.substr(0, lastDot);
    }

    // 检查名称是否为空，如果为空则使用默认名称
    if (name.empty()) {
        name = "未命名脚本";
    }

    // 从脚本内容中提取描述信息
    std::string description;
    std::regex descRegex("#脚本功能:(.*)");
    std::smatch match;
    if (std::regex_search(content, match, descRegex) && match.size() > 1) {
        description = match[1].str();
        // 去除前后空格
        description = std::regex_replace(description, std::regex("^\\s+|\\s+$"), "");
    }

    // 如果描述为空，设置默认描述
    if (description.empty()) {
        description = "";
    }

    return ParseScript(content, name, description);
}

// 从字符串解析脚本
CompiledScript ScriptParser::ParseScript(const std::string& scriptContent, const std::string& name, const std::string& description) {
    CompiledScript script;
    script.name = name;
    script.description = description;

    std::istringstream iss(scriptContent);
    std::string line;

    while (std::getline(iss, line)) {
        // 跳过空行和注释行
        if (line.empty() || line[0] == '#') {
            continue;
        }

        // 解析命令并添加到命令序列
        Command cmd = ParseCommand(line);
        if (cmd.type != CMD_INVALID) {
            script.commands.push_back(cmd);
        }
    }

    return script;
}

// 解析单行命令
Command ScriptParser::ParseCommand(const std::string& line) {
    Command cmd;
    cmd.type = CMD_INVALID;

    // 按键按下命令: 按住xxx键()
    std::regex pressKeyRegex("按住(.*?)\\(\\)");
    std::smatch pressKeyMatch;
    if (std::regex_search(line, pressKeyMatch, pressKeyRegex) && pressKeyMatch.size() > 1) {
        std::string keyName = pressKeyMatch[1].str();
        int keyCode = GetKeyCode(keyName);
        if (keyCode != 0) {
            cmd.type = CMD_PRESS_KEY;
            cmd.keyCode = keyCode;
        }
        return cmd;
    }

    // 按键释放命令: 弹起xxx键()
    std::regex releaseKeyRegex("弹起(.*?)\\(\\)");
    std::smatch releaseKeyMatch;
    if (std::regex_search(line, releaseKeyMatch, releaseKeyRegex) && releaseKeyMatch.size() > 1) {
        std::string keyName = releaseKeyMatch[1].str();
        int keyCode = GetKeyCode(keyName);
        if (keyCode != 0) {
            cmd.type = CMD_RELEASE_KEY;
            cmd.keyCode = keyCode;
        }
        return cmd;
    }

    // 睡眠命令: 睡眠(0.1)
    std::regex sleepRegex("睡眠\\((\\d+\\.?\\d*)\\)");
    std::smatch sleepMatch;
    if (std::regex_search(line, sleepMatch, sleepRegex) && sleepMatch.size() > 1) {
        cmd.type = CMD_SLEEP;
        cmd.sleepTime = std::stod(sleepMatch[1].str());
        return cmd;
    }

    return cmd;
}

// 获取按键代码
int ScriptParser::GetKeyCode(const std::string& keyName) {
    auto it = keyNameMap.find(keyName);
    if (it != keyNameMap.end()) {
        return it->second;
    }
    return 0;  // 未找到按键
}

// 执行编译后的脚本
void ScriptParser::ExecuteScript(const CompiledScript& script) {
    std::lock_guard<std::mutex> lock(g_execution_lock);
    if (g_is_executing) return;
    g_is_executing = true;

    try {
        for (const auto& cmd : script.commands) {
            switch (cmd.type) {
            case CMD_PRESS_KEY:
                SimulateKeyPress(cmd.keyCode);
                break;
            case CMD_RELEASE_KEY:
                SimulateKeyRelease(cmd.keyCode);
                break;
            case CMD_SLEEP:
                SleepFor(cmd.sleepTime);
                break;
            default:
                break;
            }
        }
    }
    catch (...) {
        // 确保出错时释放所有按键
        // 这里可以添加一个释放所有按键的函数调用
    }

    g_is_executing = false;
}

// 将脚本转换为可执行函数
std::function<void()> ScriptParser::CreateExecutableFunction(const CompiledScript& script) {
    return [script]() {
        ExecuteScript(script);
        };
}

// 脚本注册器实现
void ScriptRegistrar::LoadScriptsFromDirectory(const std::string& directory) {
    // 在加载脚本前初始化键位映射表
    ScriptParser::InitKeyNameMap();

    // 确保目录存在
    CreateDirectoryA(directory.c_str(), NULL);

    // 使用宽字符版本的API来处理中文文件名
    WIN32_FIND_DATAW findData;
    std::wstring searchPath = Utf8ToWide(directory + "\\*.txt");
    HANDLE hFind = FindFirstFileW(searchPath.c_str(), &findData);

    if (hFind == INVALID_HANDLE_VALUE) {
        return;
    }

    do {
        if (!(findData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)) {
            // 转换宽字符文件名为UTF-8
            std::string fileName = WideToUtf8(findData.cFileName);

            // 使用宽字符路径打开文件
            std::wstring wideFilePath = Utf8ToWide(directory) + L"\\" + findData.cFileName;

            try {
                // 使用宽字符API打开文件
                std::ifstream file(wideFilePath, std::ios::binary);
                if (!file.is_open()) {
                    throw std::runtime_error("无法打开脚本文件: " + fileName);
                }

                // 检测UTF-8 BOM
                char bom[3];
                bool hasBom = false;
                if (file.read(bom, 3) && bom[0] == (char)0xEF && bom[1] == (char)0xBB && bom[2] == (char)0xBF) {
                    hasBom = true;
                }
                else {
                    file.seekg(0); // 如果没有BOM，重置文件指针到开头
                }

                // 读取整个文件内容
                std::string content((std::istreambuf_iterator<char>(file)), std::istreambuf_iterator<char>());
                file.close();

                // 直接使用文件名作为脚本名称
                std::string name = fileName;
                size_t lastDot = name.find_last_of(".");
                if (lastDot != std::string::npos) {
                    name = name.substr(0, lastDot);
                }

                // 从脚本内容中提取描述信息
                std::string description;
                std::regex descRegex("#脚本功能:(.*)");
                std::smatch match;
                if (std::regex_search(content, match, descRegex) && match.size() > 1) {
                    description = match[1].str();
                    // 去除前后空格
                    description = std::regex_replace(description, std::regex("^\\s+|\\s+$"), "");
                }

                // 如果描述为空，设置默认描述
                if (description.empty()) {
                    description = "";
                }

                // 解析脚本内容 - 修正这里，添加ScriptParser::前缀
                CompiledScript script = ScriptParser::ParseScript(content, name, description);
                RegisterScript(script);
            }
            catch (const std::exception& e) {
                // 将错误信息显示为对话框而不是控制台输出
                std::string errorMsg = "加载脚本失败: " + fileName + "\n原因: " + e.what();
                MessageBoxA(NULL, errorMsg.c_str(), "脚本加载错误", MB_OK | MB_ICONERROR);
            }
        }
    } while (FindNextFileW(hFind, &findData));

    FindClose(hFind);

    // 在加载完所有脚本后保存配置
    g_script_manager.SaveScriptConfig();
}

// 注册单个脚本
void ScriptRegistrar::RegisterScript(const CompiledScript& script, int hotkey) {
    try {
        // 如果未指定热键，先尝试从现有配置中查找
        if (hotkey == 0) {
            // 查找脚本配置中是否已有该脚本的热键设置
            std::ifstream file("scripts_config.json", std::ios::in | std::ios::binary);
            if (file.is_open()) {
                // 检测并跳过UTF-8 BOM
                char bom[3];
                if (file.read(bom, 3) && !(bom[0] == (char)0xEF && bom[1] == (char)0xBB && bom[2] == (char)0xBF)) {
                    file.seekg(0); // 如果没有BOM，重置文件指针到开头
                }

                // 读取文件内容
                std::string jsonContent((std::istreambuf_iterator<char>(file)), std::istreambuf_iterator<char>());
                file.close();

                // 检查内容是否为空或仅包含空白字符
                if (!jsonContent.empty() && jsonContent.find_first_not_of(" \t\n\r") != std::string::npos) {
                    try {
                        // 解析JSON
                        json j = json::parse(jsonContent);

                        if (j.contains("scripts") && j["scripts"].is_array()) {
                            // 查找匹配的脚本名称
                            for (const auto& scriptObj : j["scripts"]) {
                                if (scriptObj.contains("name") && scriptObj["name"] == script.name) {
                                    // 找到匹配的脚本，使用其热键
                                    hotkey = scriptObj["hotkey"];
                                    break;
                                }
                            }
                        }
                    }
                    catch (const json::exception& e) {
                        // JSON解析失败，创建默认配置
                        CreateDefaultConfig();
                    }
                }
                else {
                    // 文件为空，创建默认配置
                    CreateDefaultConfig();
                }
            }
            else {
                // 文件不存在，创建默认配置
                CreateDefaultConfig();
            }

            // 如果仍未找到热键，使用默认键位 '-'
            if (hotkey == 0) {
                hotkey = VK_OEM_MINUS;  // '-' 键的虚拟键码
            }
        }

        Script newScript;
        newScript.name = script.name;
        newScript.description = script.description;
        newScript.hotkey = hotkey;
        newScript.execute = ScriptParser::CreateExecutableFunction(script);

        g_script_manager.AddScript(newScript);
    }
    catch (const std::exception& e) {
        MessageBoxA(NULL, (std::string("注册脚本失败: ") + script.name + "\n原因: " + e.what()).c_str(),
            "脚本注册错误", MB_OK | MB_ICONERROR);
    }
}

// 添加创建默认配置的函数
void ScriptRegistrar::CreateDefaultConfig() {
    try {
        json defaultConfig;
        defaultConfig["scripts"] = json::array();

        std::ofstream file("scripts_config.json", std::ios::out | std::ios::binary);
        if (file.is_open()) {
            // 写入UTF-8 BOM
            file.write("\xEF\xBB\xBF", 3);
            file << defaultConfig.dump(4);  // 使用4个空格缩进
            file.close();
        }
    }
    catch (...) {
        // 忽略创建默认配置时的错误
    }
}

// 清除所有注册的脚本
void ScriptRegistrar::ClearScripts() {
    // 调用ScriptManager的ClearScripts方法
    g_script_manager.ClearScripts();
}
