﻿#pragma execution_character_set("utf-8")
#pragma comment(linker, "/subsystem:console")  // 修改为控制台程序，以便查看输出
#pragma comment(lib, "d3d11.lib")
#pragma comment(lib, "dxgi.lib")
#pragma comment(lib, "d3dcompiler.lib")
#pragma comment(lib, "ws2_32.lib")  // 添加Winsock库链接

// 在所有头文件之前定义这些宏，防止Winsock冲突
#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif
#ifndef NOMINMAX
#define NOMINMAX
#endif
// 排除Winsock1，只使用Winsock2
#define _WINSOCKAPI_

// 首先包含Winsock2头文件（必须在任何Windows头文件之前）
#include <winsock2.h>
#include <ws2tcpip.h>

#include <Windows.h>
#include <string>
#include <thread>
#include <mutex>
#include <iostream>
#include <sstream>
#include <atomic>
#include <chrono>
#include <iomanip>
#include <locale>
#include <io.h>
#include <fcntl.h>
#include "imgui/imgui.h"
#include "imgui/imgui_impl_win32.h"
#include "imgui/imgui_impl_dx11.h"
#include <d3d11.h>
#include "nba2kol2_keys.h"
#include "script_functions.h"
#include "script_parser.h"
#include "nba2kol2_keys.h"
#include "license_verifier.h"
#include "license_window.h"
// AI智能脚本相关全局变量
static bool g_ai_script_enabled = false;  // AI智能脚本开关
static std::string g_current_region = "弧顶";  // 当前区域信息
static std::mutex g_region_mutex;  // 区域信息互斥锁
// 在文件开头添加区域编码映射
struct RegionMapping {
    static const int ARC_TOP = 1;           // 弧顶
    static const int LEFT_LOW_CORNER = 2;   // 左低角
    static const int RIGHT_LOW_CORNER = 3;  // 右低角
    static const int ARC_TOP_LEFT_45 = 4;   // 弧顶左45度
    static const int ARC_TOP_RIGHT_45 = 5;  // 弧顶右45度

    // 编码到区域名称的映射
    static std::string getRegionName(int code) {
        switch (code) {
        case ARC_TOP: return "弧顶";
        case LEFT_LOW_CORNER: return "左底角";
        case RIGHT_LOW_CORNER: return "右底角";
        case ARC_TOP_LEFT_45: return "左侧";
        case ARC_TOP_RIGHT_45: return "右侧";
        default: return "弧顶";
        }
    }

    // 编码到描述信息的映射
    static std::string getRegionDescription(int code) {
        switch (code) {
        case ARC_TOP: return "当前位置: 三分线弧顶区域 - 适合远距离投篮";
        case LEFT_LOW_CORNER: return "当前位置: 左侧低位角落 - 适合角度投篮";
        case RIGHT_LOW_CORNER: return "当前位置: 右侧低位角落 - 适合角度投篮";
        case ARC_TOP_LEFT_45: return "当前位置: 弧顶左侧45度 - 适合侧翼投篮";
        case ARC_TOP_RIGHT_45: return "当前位置: 弧顶右侧45度 - 适合侧翼投篮";
        default: return "当前位置: 其他区域 - 位置待确定";
        }
    }
};

// 添加Socket客户端类
class RegionSocketClient {
private:
    SOCKET clientSocket;
    std::thread clientThread;
    std::atomic<bool> running{ false };
    std::atomic<bool> connected{ false };
    std::string serverIP;
    int serverPort;
    std::mutex receiveMutex;
    std::string lastReceivedData;

public:
    RegionSocketClient(const std::string& ip = "127.0.0.1", int port = 12345)
        : serverIP(ip), serverPort(port), clientSocket(INVALID_SOCKET) {
        // 初始化Winsock
        WSADATA wsaData;
        int result = WSAStartup(MAKEWORD(2, 2), &wsaData);
        if (result != 0) {
            std::cerr << "WSAStartup失败: " << result << std::endl;
            return;
        }
        std::cout << "Socket客户端初始化完成" << std::endl;
    }

    ~RegionSocketClient() {
        Stop();
        WSACleanup();
    }

    bool Start() {
        // 创建Socket
        clientSocket = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
        if (clientSocket == INVALID_SOCKET) {
            std::cerr << "Failed to create client socket: " << WSAGetLastError() << std::endl;
            return false;
        }

        // 设置Socket为非阻塞模式
        u_long mode = 1;
        ioctlsocket(clientSocket, FIONBIO, &mode);

        running = true;
        clientThread = std::thread(&RegionSocketClient::ClientLoop, this);

        std::cout << "Socket client started, trying to connect to: " << serverIP << ":" << serverPort << std::endl;
        return true;
    }

    void Stop() {
        running = false;
        connected = false;

        if (clientThread.joinable()) {
            clientThread.join();
        }

        if (clientSocket != INVALID_SOCKET) {
            closesocket(clientSocket);
            clientSocket = INVALID_SOCKET;
        }
    }

    bool IsConnected() const {
        return connected.load();
    }

    std::string GetLastReceivedData() {
        std::lock_guard<std::mutex> lock(receiveMutex);
        return lastReceivedData;
    }

private:
    void ClientLoop() {
        while (running) {
            if (!connected) {
                // 尝试连接服务器
                if (TryConnect()) {
                    connected = true;
                    std::cout << "[SUCCESS] Connected to basketball AI detection server!" << std::endl;
                    std::cout << "Listening for region data..." << std::endl;
                    std::cout << "========================================" << std::endl;
                }
                else {
                    // 连接失败，等待一段时间后重试
                    std::this_thread::sleep_for(std::chrono::seconds(3));
                    continue;
                }
            }

            // 尝试接收数据
            if (connected && !ReceiveData()) {
                // 接收失败，可能断开连接
                connected = false;
                std::cout << "[ERROR] Connection lost, trying to reconnect..." << std::endl;

                // 关闭当前socket
                closesocket(clientSocket);
                clientSocket = INVALID_SOCKET;

                // 重新创建socket
                clientSocket = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
                if (clientSocket != INVALID_SOCKET) {
                    u_long mode = 1;
                    ioctlsocket(clientSocket, FIONBIO, &mode);
                }
            }

            // 短暂休眠，避免CPU占用过高
            std::this_thread::sleep_for(std::chrono::milliseconds(100));
        }
    }

    bool TryConnect() {
        if (clientSocket == INVALID_SOCKET) {
            return false;
        }

        // 设置服务器地址
        sockaddr_in serverAddr;
        ZeroMemory(&serverAddr, sizeof(serverAddr));
        serverAddr.sin_family = AF_INET;
        serverAddr.sin_port = htons(static_cast<u_short>(serverPort));

        // 转换IP地址
        if (inet_pton(AF_INET, serverIP.c_str(), &serverAddr.sin_addr) <= 0) {
            std::cerr << "Invalid IP address: " << serverIP << std::endl;
            return false;
        }

        // 尝试连接（非阻塞模式）
        int result = connect(clientSocket, reinterpret_cast<const sockaddr*>(&serverAddr), sizeof(serverAddr));

        if (result == SOCKET_ERROR) {
            int error = WSAGetLastError();
            if (error == WSAEWOULDBLOCK || error == WSAEALREADY) {
                // 连接正在进行中，等待一下
                std::this_thread::sleep_for(std::chrono::milliseconds(500));

                // 检查连接状态
                fd_set writeSet;
                FD_ZERO(&writeSet);
                FD_SET(clientSocket, &writeSet);

                timeval timeout;
                timeout.tv_sec = 1;
                timeout.tv_usec = 0;

                int selectResult = select(0, nullptr, &writeSet, nullptr, &timeout);
                if (selectResult > 0 && FD_ISSET(clientSocket, &writeSet)) {
                    // 检查连接是否真的成功
                    int error = 0;
                    int errorLen = sizeof(error);
                    if (getsockopt(clientSocket, SOL_SOCKET, SO_ERROR, reinterpret_cast<char*>(&error), &errorLen) == 0 && error == 0) {
                        return true;
                    }
                }
                return false;
            }
            else if (error == WSAEISCONN) {
                // 已经连接
                return true;
            }
            else {
                // 其他错误
                std::cerr << "Failed to connect to server: " << error << std::endl;
                return false;
            }
        }

        return true;
    }

    bool ReceiveData() {
        char buffer[1024];
        int bytesReceived = recv(clientSocket, buffer, sizeof(buffer) - 1, 0);

        if (bytesReceived > 0) {
            buffer[bytesReceived] = '\0';
            std::string receivedData(buffer);

            // 解析JSON数据并输出
            ParseAndDisplayData(receivedData);

            // 保存最后接收的数据
            {
                std::lock_guard<std::mutex> lock(receiveMutex);
                lastReceivedData = receivedData;
            }

            return true;
        }
        else if (bytesReceived == 0) {
            // 服务器关闭连接
            std::cout << "Server closed the connection" << std::endl;
            return false;
        }
        else {
            int error = WSAGetLastError();
            if (error != WSAEWOULDBLOCK) {
                std::cerr << "Failed to receive data: " << error << std::endl;
                return false;
            }
            // WSAEWOULDBLOCK 表示没有数据可读，这是正常的
            return true;
        }
    }

    void ParseAndDisplayData(const std::string& jsonData) {
        try {
            // 简单的JSON解析（手动解析关键字段）
            std::string timestamp, type;
            int regionCode = 1; // 默认为弧顶

            // 查找type字段
            auto findStringValue = [](const std::string& json, const std::string& key) -> std::string {
                std::string searchKey = "\"" + key + "\":\"";
                size_t pos = json.find(searchKey);
                if (pos != std::string::npos) {
                    pos += searchKey.length();
                    size_t endPos = json.find("\"", pos);
                    if (endPos != std::string::npos) {
                        return json.substr(pos, endPos - pos);
                    }
                }
                return "";
                };

            // 查找数字字段
            auto findIntValue = [](const std::string& json, const std::string& key) -> int {
                std::string searchKey = "\"" + key + "\":";
                size_t pos = json.find(searchKey);
                if (pos != std::string::npos) {
                    pos += searchKey.length();
                    size_t endPos = json.find_first_of(",}", pos);
                    if (endPos != std::string::npos) {
                        std::string valueStr = json.substr(pos, endPos - pos);
                        try {
                            return std::stoi(valueStr);
                        }
                        catch (...) {
                            return 1; // 默认值
                        }
                    }
                }
                return 1; // 默认值
                };

            type = findStringValue(jsonData, "type");
            regionCode = findIntValue(jsonData, "region_code");
            timestamp = findStringValue(jsonData, "timestamp");

            // 将区域编码转换为中文名称
            std::string regionName = RegionMapping::getRegionName(regionCode);
            std::string regionDescription = RegionMapping::getRegionDescription(regionCode);

            // 更新全局区域信息
            {
                std::lock_guard<std::mutex> lock(g_region_mutex);
                g_current_region = regionName;
            }

            // 获取当前时间用于显示
            auto now = std::chrono::system_clock::now();
            auto time_t = std::chrono::system_clock::to_time_t(now);
            auto ms = std::chrono::duration_cast<std::chrono::milliseconds>(
                now.time_since_epoch()) % 1000;

            char timeBuffer[100];
            struct tm timeinfo;
            localtime_s(&timeinfo, &time_t);
            strftime(timeBuffer, sizeof(timeBuffer), "%H:%M:%S", &timeinfo);

            // 格式化输出（使用解码后的中文）
            std::cout << "[" << timeBuffer << "." << std::setfill('0') << std::setw(3) << ms.count() << "] "
                << "📍 篮球区域更新: " << regionName << " (编码:" << regionCode << ")";

            if (!timestamp.empty()) {
                std::cout << " | 服务器时间: " << timestamp;
            }

            std::cout << std::endl;

            // 输出描述信息
            std::cout << "   ➤ " << regionDescription << std::endl;
            std::cout << "----------------------------------------" << std::endl;
            std::cout.flush(); // 强制刷新输出缓冲区

        }
        catch (const std::exception& e) {
            std::cerr << "解析JSON数据时出错: " << e.what() << std::endl;
            std::cerr << "原始数据: " << jsonData << std::endl;
        }
    }
};

// 添加Socket客户端全局变量
static std::unique_ptr<RegionSocketClient> g_socket_client = nullptr;

// 全局变量
bool g_tool_enabled = true;  // 修改为默认启用
bool g_is_executing = false;
bool g_show_window = true;  // 添加窗口显示控制变量
// bool g_show_status_window = true;  // 删除状态窗口显示控制变量
std::mutex g_execution_lock;

// 授权相关全局状态
static bool g_license_verified = false;  // 新增：是否已通过验证



// 添加窗口句柄全局变量
HWND g_main_hwnd = nullptr;
// HWND g_status_hwnd = nullptr;  // 删除状态窗口句柄

// 在全局变量区域添加
int* g_selected_key = nullptr;  // 当前正在设置的键位指针
bool g_is_listening_for_key = false;  // 是否正在监听键位
static bool g_main_window_listening = false;  // 新增：窗口类是否正在监听按键

// 删除Config结构体中的触发键
struct Config {
    // int trigger_key = 0x52;   // 删除触发键R
} g_config;

// 全局配置对象
extern NBA2KOL2Keys g_keys;

extern IMGUI_IMPL_API LRESULT ImGui_ImplWin32_WndProcHandler(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam);

// 窗口过程函数
LRESULT WINAPI WndProc(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam) {
    if (ImGui_ImplWin32_WndProcHandler(hWnd, msg, wParam, lParam))
        return true;

    switch (msg) {
    case WM_SIZE:
        return 0;
    case WM_SYSCOMMAND:
        if ((wParam & 0xfff0) == SC_KEYMENU)
            return 0;
        break;
    case WM_DESTROY:
        PostQuitMessage(0);
        return 0;
    }
    return DefWindowProc(hWnd, msg, wParam, lParam);
}

// 添加键位监听函数
void StartListeningForKey(int* key_ptr) {
    g_selected_key = key_ptr;
    g_is_listening_for_key = true;
}

// 添加AI智能脚本执行函数
void ExecuteAIScript(int hotkey) {
    if (!g_ai_script_enabled) {
        // AI智能脚本未开启，执行原始逻辑
        const auto& scripts = g_script_manager.GetScripts();
        for (const auto& script : scripts) {
            if (script.hotkey == hotkey) {
                script.execute();
                std::cout << "执行脚本: " << script.name << std::endl;
                break;
            }
        }
        return;
    }

    // AI智能脚本已开启，根据当前区域匹配脚本
    std::string currentRegion;
    {
        std::lock_guard<std::mutex> lock(g_region_mutex);
        currentRegion = g_current_region;
    }

    const auto& scripts = g_script_manager.GetScripts();
    std::vector<const Script*> candidateScripts;

    // 找到所有绑定到该热键的脚本
    for (const auto& script : scripts) {
        if (script.hotkey == hotkey) {
            candidateScripts.push_back(&script);
        }
    }

    if (candidateScripts.empty()) {
        std::cout << "❌ 未找到绑定到热键的脚本" << std::endl;
        return;
    }

    // 如果只有一个脚本，直接执行
    if (candidateScripts.size() == 1) {
        candidateScripts[0]->execute();
        std::cout << "🎯 执行脚本: " << candidateScripts[0]->name << " (唯一脚本)" << std::endl;
        return;
    }

    // 多个脚本时，根据当前区域进行精确匹配
    const Script* bestMatch = nullptr;
    int bestMatchScore = 0;  // 匹配得分，得分越高越精确

    std::cout << "🤖 AI智能脚本匹配中..." << std::endl;
    std::cout << "当前区域: " << currentRegion << std::endl;
    std::cout << "候选脚本(" << candidateScripts.size() << "个):" << std::endl;

    for (const auto* script : candidateScripts) {
        std::cout << "  - " << script->name << std::endl;

        int matchScore = 0;

        // 1. 首先尝试完全匹配
        if (script->name.find(currentRegion) != std::string::npos) {
            matchScore = currentRegion.length();  // 匹配长度作为得分

            // 2. 检查是否为精确匹配（避免部分匹配）
            // 例如："弧顶左45度" 应该优先于 "弧顶"
            size_t pos = script->name.find(currentRegion);
            if (pos != std::string::npos) {
                // 检查匹配位置前后的字符，确保是完整的词汇匹配
                bool isCompleteMatch = true;

                // 检查前一个字符
                if (pos > 0) {
                    char prevChar = script->name[pos - 1];
                    if (isalnum(prevChar) || prevChar > 127) {  // 中文字符或字母数字
                        // 如果前面是字母数字或中文，可能是部分匹配
                        isCompleteMatch = false;
                    }
                }

                // 检查后一个字符
                if (pos + currentRegion.length() < script->name.length()) {
                    char nextChar = script->name[pos + currentRegion.length()];
                    if (isalnum(nextChar) || nextChar > 127) {  // 中文字符或字母数字
                        // 如果后面是字母数字或中文，可能是部分匹配
                        isCompleteMatch = false;
                    }
                }

                if (isCompleteMatch) {
                    matchScore += 100;  // 完整匹配加分
                }

                std::cout << "    ✓ 匹配得分: " << matchScore << " (区域长度:" << currentRegion.length() << ")" << std::endl;
            }
        }

        // 更新最佳匹配
        if (matchScore > bestMatchScore) {
            bestMatchScore = matchScore;
            bestMatch = script;
        }
    }

    if (bestMatch && bestMatchScore > 0) {
        bestMatch->execute();
        std::cout << "🎯 智能执行脚本: " << bestMatch->name << " (得分:" << bestMatchScore << ")" << std::endl;
    }
    else {
        // 没有匹配的脚本，执行第一个
        candidateScripts[0]->execute();
        std::cout << "⚠️ 未找到区域匹配脚本，执行默认脚本: " << candidateScripts[0]->name << std::endl;
    }

    std::cout << "========================================" << std::endl;
}

// 添加精确匹配算法
int CalculateMatchScore(const std::string& scriptName, const std::string& region) {
    // 如果脚本名中不包含区域名，得分为0
    size_t pos = scriptName.find(region);
    if (pos == std::string::npos) {
        return 0;
    }

    int score = 0;

    // 基础得分：区域名长度
    score += region.length() * 10;

    // 精确匹配奖励：检查区域名前后是否为独立词汇
    bool isWordBoundary = true;

    // 检查前边界
    if (pos > 0) {
        char prevChar = scriptName[pos - 1];
        // 如果前面是中文字符、字母或数字，认为不是独立词汇
        if ((prevChar >= 'a' && prevChar <= 'z') ||
            (prevChar >= 'A' && prevChar <= 'Z') ||
            (prevChar >= '0' && prevChar <= '9') ||
            (unsigned char)prevChar > 127) {  // 中文字符
            isWordBoundary = false;
        }
    }

    // 检查后边界
    if (pos + region.length() < scriptName.length()) {
        char nextChar = scriptName[pos + region.length()];
        // 如果后面是中文字符、字母或数字，认为不是独立词汇
        if ((nextChar >= 'a' && nextChar <= 'z') ||
            (nextChar >= 'A' && nextChar <= 'Z') ||
            (nextChar >= '0' && nextChar <= '9') ||
            (unsigned char)nextChar > 127) {  // 中文字符
            isWordBoundary = false;
        }
    }

    // 如果是独立词汇，给予高分奖励
    if (isWordBoundary) {
        score += 1000;  // 独立词汇高分奖励
    }
    else {
        score += 1;     // 部分匹配低分
    }

    return score;
}

// 修改按键监听线程
void MonitorThread() {
    while (true) {
        // 检查Home键是否被按下
        if (GetAsyncKeyState(VK_HOME) & 0x8000) {
            g_show_window = !g_show_window;  // 切换窗口显示状态

            // 控制整个Windows窗口的显示/隐藏
            if (g_main_hwnd != nullptr) {
                if (g_show_window) {
                    ShowWindow(g_main_hwnd, SW_SHOW);  // 显示窗口
                    SetForegroundWindow(g_main_hwnd);  // 将窗口置于前台
                }
                else {
                    ShowWindow(g_main_hwnd, SW_HIDE);  // 隐藏窗口
                }
            }

            Sleep(300);  // 防止连续触发
        }

        // 检查"0"键是否被按下（作为工具开关）
        if (GetAsyncKeyState('0') & 0x8000) {  // '0' 是主键盘 "0" 键
            if (g_license_verified) {
                g_tool_enabled = !g_tool_enabled;  // 切换工具状态
            }
            Sleep(300);  // 防止连续触发
        }

        // 检查"="键是否被按下（作为AI智能脚本开关）
        if (GetAsyncKeyState(VK_OEM_PLUS) & 0x8000) {  // VK_OEM_PLUS 是 "=" 键
            if (g_license_verified) {
                g_ai_script_enabled = !g_ai_script_enabled;  // 切换AI智能脚本状态
                std::cout << "🤖 AI智能脚本: " << (g_ai_script_enabled ? "已开启" : "已关闭") << std::endl;
            }
            Sleep(300);  // 防止连续触发
        }

        // 监听新键位设置（旧系统，用于非脚本热键）
        if (g_is_listening_for_key && g_selected_key != nullptr) {
            for (int i = 0x01; i <= 0xFE; i++) {
                if (GetAsyncKeyState(i) & 0x8000) {
                    *g_selected_key = i;
                    g_is_listening_for_key = false;
                    g_selected_key = nullptr;

                    // 键位修改完成后自动加载脚本
                    try {
                        // 保存当前脚本配置
                        g_script_manager.SaveScriptConfig();

                        // 重新加载脚本目录中的所有脚本
                        ScriptRegistrar::ClearScripts();
                        ScriptRegistrar::LoadScriptsFromDirectory("scripts");

                        // 加载完成后再次保存配置，确保热键信息正确
                        g_script_manager.SaveScriptConfig();
                    }
                    catch (const std::exception& e) {
                        // 静默处理错误，不显示错误信息
                    }

                    Sleep(300);  // 防止连续触发
                    break;
                }
            }
        }

        // 关键修改：只有在没有窗口类按键监听时才执行脚本热键检测
        if (g_license_verified && g_tool_enabled && !g_is_listening_for_key && !g_main_window_listening) {
            // 检查所有注册脚本的热键 - 修改：只检测设置了热键的脚本
            const auto& scripts = g_script_manager.GetScripts();
            for (const auto& script : scripts) {
                if (script.hotkey != 0 && GetAsyncKeyState(script.hotkey) & 0x8000) {
                    // 使用新的AI智能脚本执行函数
                    ExecuteAIScript(script.hotkey);

                    break;
                }
            }
        }
        Sleep(1);
    }
}

// 主窗口类
class MainWindow {
private:
    ID3D11Device* m_pd3dDevice = nullptr;
    ID3D11DeviceContext* m_pd3dDeviceContext = nullptr;
    IDXGISwapChain* m_pSwapChain = nullptr;
    ID3D11RenderTargetView* m_mainRenderTargetView = nullptr;
    HWND m_hwnd = nullptr;

    // 添加按键监听相关的成员变量
    std::thread m_key_listening_thread;
    bool m_is_listening_for_key = false;
    int* m_target_key_code = nullptr;
    std::mutex m_key_listening_mutex;

    size_t m_is_listening_for_script_index = -1;

    // 添加标签搜索相关的成员变量
    int m_selected_tag_index;  // 当前选择的标签索引
    bool m_show_only_bound_scripts = false;  // 只显示已绑定热键的脚本

    // 在MainWindow类中添加热键冲突检测函数
    bool IsHotkeyConflict(int hotkey) {
        if (hotkey == 0) return false; // 0表示未设置，不算冲突

        // 检查基础操作键位
        if (hotkey == g_keys.shoot_key ||
            hotkey == g_keys.post_up_key ||
            hotkey == g_keys.pass_key ||
            hotkey == g_keys.sprint_key ||
            hotkey == g_keys.bounce_pass_key ||
            hotkey == g_keys.lob_pass_key ||
            hotkey == g_keys.post_move_key) {
            return true;
        }

        // 检查移动方向键
        if (hotkey == g_keys.move_up_key ||
            hotkey == g_keys.move_down_key ||
            hotkey == g_keys.move_left_key ||
            hotkey == g_keys.move_right_key) {
            return true;
        }

        // 检查特殊方向键
        if (hotkey == g_keys.special_up_key ||
            hotkey == g_keys.special_down_key ||
            hotkey == g_keys.special_left_key ||
            hotkey == g_keys.special_right_key) {
            return true;
        }

        // 检查斜向特殊方向键
        if (hotkey == g_keys.special_left_up_key ||
            hotkey == g_keys.special_left_down_key ||
            hotkey == g_keys.special_right_down_key ||
            hotkey == g_keys.special_right_up_key) {
            return true;
        }

        // 检查其他特殊键
        if (hotkey == g_keys.right_hand_spin_key ||
            hotkey == g_keys.left_hand_spin_key ||
            hotkey == g_keys.right_hand_turn_key ||
            hotkey == g_keys.left_hand_turn_key) {
            return true;
        }

        return false;
    }

    // 获取冲突的键位名称
    std::string GetConflictKeyName(int hotkey) {
        if (hotkey == g_keys.shoot_key) return "投篮键";
        if (hotkey == g_keys.post_up_key) return "背身键";
        if (hotkey == g_keys.pass_key) return "传球键";
        if (hotkey == g_keys.sprint_key) return "冲刺键";
        if (hotkey == g_keys.bounce_pass_key) return "击地传球键";
        if (hotkey == g_keys.lob_pass_key) return "高吊传球键";
        if (hotkey == g_keys.post_move_key) return "背身单打键";

        if (hotkey == g_keys.move_up_key) return "向上移动键";
        if (hotkey == g_keys.move_down_key) return "向下移动键";
        if (hotkey == g_keys.move_left_key) return "向左移动键";
        if (hotkey == g_keys.move_right_key) return "向右移动键";

        if (hotkey == g_keys.special_up_key) return "特殊上键";
        if (hotkey == g_keys.special_down_key) return "特殊下键";
        if (hotkey == g_keys.special_left_key) return "特殊左键";
        if (hotkey == g_keys.special_right_key) return "特殊右键";

        if (hotkey == g_keys.special_left_up_key) return "特殊方向左上键";
        if (hotkey == g_keys.special_left_down_key) return "特殊方向左下键";
        if (hotkey == g_keys.special_right_down_key) return "特殊方向右下键";
        if (hotkey == g_keys.special_right_up_key) return "特殊方向右上键";

        if (hotkey == g_keys.right_hand_spin_key) return "右手持球半转身键";
        if (hotkey == g_keys.left_hand_spin_key) return "左手持球半转身键";
        if (hotkey == g_keys.right_hand_turn_key) return "右手持球转身键";
        if (hotkey == g_keys.left_hand_turn_key) return "左手持球转身键";

        return "未知键位";
    }

    // 在MainWindow类中添加字符串转换辅助函数
    std::wstring Utf8ToWide(const std::string& utf8Str) {
        if (utf8Str.empty()) return std::wstring();

        int size_needed = MultiByteToWideChar(CP_UTF8, 0, &utf8Str[0], (int)utf8Str.size(), NULL, 0);
        std::wstring wstrTo(size_needed, 0);
        MultiByteToWideChar(CP_UTF8, 0, &utf8Str[0], (int)utf8Str.size(), &wstrTo[0], size_needed);
        return wstrTo;
    }

    void StartKeyListeningThread(int* target_key_code) {
        try {
            std::lock_guard<std::mutex> lock(m_key_listening_mutex);
            if (m_is_listening_for_key) {
                StopKeyListeningThread();
            }

            // 检查指针是否有效
            if (target_key_code == nullptr) {
                throw std::runtime_error("无效的按键指针");
            }

            m_is_listening_for_key = true;
            m_target_key_code = target_key_code;
            g_main_window_listening = true;  // 新增：设置全局标志

            m_key_listening_thread = std::thread([this]() {
                try {
                    // 等待当前按键释放
                    Sleep(200);

                    // 记录按键状态
                    bool key_states[256] = { false };
                    for (int i = 0x08; i <= 0xFE; i++) {
                        key_states[i] = (GetAsyncKeyState(i) & 0x8000) != 0;
                    }

                    while (m_is_listening_for_key) {
                        bool key_detected = false;

                        for (int i = 0x08; i <= 0xFE; i++) {
                            bool current_state = (GetAsyncKeyState(i) & 0x8000) != 0;

                            // 只有当键从未按下状态变为按下状态时才触发
                            if (current_state && !key_states[i]) {
                                std::lock_guard<std::mutex> lock(m_key_listening_mutex);
                                if (m_target_key_code != nullptr) {
                                    // 检查是否为脚本热键设置，如果是则检查冲突
                                    if (m_is_listening_for_script_index != -1) {
                                        if (IsHotkeyConflict(i)) {
                                            // 显示冲突提示 - 使用MessageBoxW支持中文
                                            std::string conflictMsg = "热键与键位设置冲突：\n" +
                                                GetKeyName(i) + " 已被 " +
                                                GetConflictKeyName(i) + " 使用\n" +
                                                "请选择其他按键";
                                            MessageBoxW(NULL, Utf8ToWide(conflictMsg).c_str(),
                                                L"热键冲突", MB_OK | MB_ICONWARNING);
                                            // 不设置热键，继续监听
                                            key_states[i] = current_state;
                                            continue;
                                        }
                                    }

                                    *m_target_key_code = i;

                                    // 判断是否为脚本热键设置
                                    if (m_is_listening_for_script_index != -1) {
                                        g_script_manager.SaveScriptConfig();  // 保存脚本配置
                                    }
                                    else {
                                        SaveKeysConfig();  // 保存键位配置
                                    }

                                    m_is_listening_for_key = false;
                                    key_detected = true;
                                    break;
                                }
                            }

                            // 更新键状态
                            key_states[i] = current_state;
                        }

                        if (key_detected) break;
                        Sleep(10);  // 降低CPU使用率
                    }
                }
                catch (const std::exception& e) {
                    std::wstring errorMsg = L"按键监听线程错误: " +
                        std::wstring(e.what(), e.what() + strlen(e.what()));
                    MessageBoxW(NULL, errorMsg.c_str(),
                        L"按键监听错误", MB_OK | MB_ICONERROR);
                    m_is_listening_for_key = false;
                }
                });
            m_key_listening_thread.detach();
        }
        catch (const std::exception& e) {
            std::wstring errorMsg = L"启动按键监听线程失败: " +
                std::wstring(e.what(), e.what() + strlen(e.what()));
            MessageBoxW(NULL, errorMsg.c_str(),
                L"按键监听错误", MB_OK | MB_ICONERROR);
            m_is_listening_for_key = false;
        }
    }

    void StopKeyListeningThread() {
        std::lock_guard<std::mutex> lock(m_key_listening_mutex);
        if (m_is_listening_for_key) {
            m_is_listening_for_key = false;
            m_target_key_code = nullptr;
            g_main_window_listening = false;  // 新增：清除全局标志
            if (m_key_listening_thread.joinable()) {
                m_key_listening_thread.join();
            }
        }
    }

    // 在MainWindow类中添加预设切换相关的方法
    void ShowPresetButtons() {
        // 添加预设选择按钮
        if (ImGui::Button("OL 预设")) {
            SwitchKeyPreset(KeyPreset::OL_DEFAULT);
        }
        ImGui::SameLine();
        if (ImGui::Button("2K 预设")) {
            SwitchKeyPreset(KeyPreset::K2_DEFAULT);
        }
        ImGui::SameLine();
        ImGui::TextDisabled("(?)");
        if (ImGui::IsItemHovered()) {
            ImGui::BeginTooltip();
            ImGui::PushTextWrapPos(ImGui::GetFontSize() * 35.0f);
            ImGui::TextUnformatted("OL Default Preset: Original key bindings");
            ImGui::TextUnformatted("2K Default Preset: WASD movement, Space pass, Numeric keys on keypad, etc.");
            ImGui::PopTextWrapPos();
            ImGui::EndTooltip();
        }
    }

    // 添加标签过滤相关的方法
    void ShowScriptTagFilter() {
        ImGui::Text("脚本标签筛选:");
        ImGui::SameLine();

        // 标签选择下拉框
        const char* tag_items[] = { "全部", "大背运", "魔术爆发", "其他" };
        ImGui::SetNextItemWidth(120.0f);
        if (ImGui::Combo("##TagFilter", &m_selected_tag_index, tag_items, IM_ARRAYSIZE(tag_items))) {
            // 标签选择改变时的处理
        }

        ImGui::SameLine();
        ImGui::TextDisabled("(选择标签过滤脚本)");

        // 添加只显示已绑定热键脚本的复选框
        ImGui::Checkbox("只显示已绑定热键的脚本", &m_show_only_bound_scripts);
    }

    // 检查脚本名称是否包含指定标签
    bool ScriptMatchesTag(const std::string& scriptName, int tagIndex) {
        if (tagIndex == 0) return true;  // "全部"标签显示所有脚本

        switch (tagIndex) {
        case 1: return scriptName.find("大背运") != std::string::npos;
        case 2: return scriptName.find("魔术爆发") != std::string::npos;
        case 3:
            // "其他"标签显示不包含指定标签的脚本
            return scriptName.find("大背运") == std::string::npos &&
                scriptName.find("魔术爆发") == std::string::npos;
        default: return true;
        }
    }

    // 获取过滤后的脚本列表
    std::vector<size_t> GetFilteredScriptIndices(const std::vector<Script>& scripts) {
        std::vector<size_t> filteredIndices;

        for (size_t i = 0; i < scripts.size(); i++) {
            const auto& script = scripts[i];

            // 标签过滤
            if (!ScriptMatchesTag(script.name, m_selected_tag_index)) {
                continue;
            }

            // 热键绑定过滤：0表示未绑定，'-'键也视为未绑定
            if (m_show_only_bound_scripts && (script.hotkey == 0 || script.hotkey == VK_OEM_MINUS)) {
                continue;
            }

            filteredIndices.push_back(i);
        }

        return filteredIndices;
    }

public:
    bool Initialize(HINSTANCE hInstance) {
        // 初始化成员变量
        m_selected_tag_index = 0;  // 默认选择"全部"

        // 加载键位配置
        LoadKeysConfig();

        // 注册窗口类
        WNDCLASSEXW wc = { sizeof(wc) };
        wc.style = CS_CLASSDC;
        wc.lpfnWndProc = WndProc;
        wc.hInstance = hInstance;
        wc.lpszClassName = L"MagicBurstTool";
        RegisterClassExW(&wc);

        // 创建主窗口
        m_hwnd = CreateWindowW(wc.lpszClassName, L"MagicBurstTool",
            WS_OVERLAPPEDWINDOW, 100, 100, 1300, 400,
            nullptr, nullptr, wc.hInstance, nullptr);

        // 保存窗口句柄到全局变量
        g_main_hwnd = m_hwnd;

        // 创建状态窗口
        // g_status_hwnd = CreateWindowW(wc.lpszClassName, L"工具状态",
        //     WS_OVERLAPPEDWINDOW, 0, 0, 400, 400,  // 左上角，400*400
        //     nullptr, nullptr, wc.hInstance, nullptr);

        if (!CreateDeviceD3D(m_hwnd)) {
            CleanupDeviceD3D();
            UnregisterClassW(wc.lpszClassName, wc.hInstance);
            return false;
        }

        ShowWindow(m_hwnd, SW_SHOWDEFAULT);
        UpdateWindow(m_hwnd);

        // ShowWindow(g_status_hwnd, SW_SHOWDEFAULT);
        // UpdateWindow(g_status_hwnd);

        // 初始化ImGui
        IMGUI_CHECKVERSION();
        ImGui::CreateContext();
        ImGuiIO& io = ImGui::GetIO();
        io.ConfigFlags |= ImGuiConfigFlags_NavEnableKeyboard;

        // 加载支持中文字符的字体
        io.Fonts->AddFontFromFileTTF("C:\\Windows\\Fonts\\msyh.ttc", 18.0f, NULL, io.Fonts->GetGlyphRangesChineseFull());

        ImGui::StyleColorsDark();
        ImGui_ImplWin32_Init(m_hwnd);
        ImGui_ImplDX11_Init(m_pd3dDevice, m_pd3dDeviceContext);

        // 加载脚本目录中的所有脚本
        ScriptRegistrar::LoadScriptsFromDirectory("scripts");

        return true;
    }

    void RenderFrame() {
        ImGui_ImplDX11_NewFrame();
        ImGui_ImplWin32_NewFrame();
        ImGui::NewFrame();

        // 授权未通过时，仅渲染授权窗口
        if (!g_license_verified) {
            static LicenseWindow licenseWindow;  // 授权窗口实例
            licenseWindow.Render();
            if (licenseWindow.IsVerified()) {
                g_license_verified = true;
            }

            // 渲染并返回，不显示主界面
            ImGui::Render();
            const float clear_color_with_alpha[4] = { 0.45f, 0.55f, 0.60f, 1.00f };
            m_pd3dDeviceContext->OMSetRenderTargets(1, &m_mainRenderTargetView, nullptr);
            m_pd3dDeviceContext->ClearRenderTargetView(m_mainRenderTargetView, clear_color_with_alpha);
            ImGui_ImplDX11_RenderDrawData(ImGui::GetDrawData());
            m_pSwapChain->Present(1, 0);
            return;
        }

        // 删除状态窗口渲染调用
        // RenderStatusWindow();

        // 只有当g_show_window为true时才渲染主UI
        if (g_show_window) {
            // 渲染主窗口
            ImGui::SetNextWindowPos(ImVec2(0, 0));
            ImGui::SetNextWindowSize(ImGui::GetIO().DisplaySize);
            ImGui::Begin("MagicBurstTool", nullptr,
                ImGuiWindowFlags_NoMove |
                ImGuiWindowFlags_NoResize |
                ImGuiWindowFlags_NoCollapse);

            // 创建左右两个面板
            float windowWidth = ImGui::GetWindowWidth();
            float leftPanelWidth = windowWidth * 0.3f;

            // 左侧面板 - 状态和基本控制
            ImGui::BeginChild("LeftPanel", ImVec2(leftPanelWidth, 0), true);

            // 工具开关
            if (ImGui::Checkbox("启用工具", &g_tool_enabled)) {
                // 状态改变时的处理
            }

            ImGui::Separator();

            // 添加状态信息显示
            ImGui::Text("工具状态:");
            if (g_tool_enabled) {
                ImGui::TextColored(ImVec4(0.0f, 1.0f, 0.0f, 1.0f), "已启动");
            }
            else {
                ImGui::TextColored(ImVec4(1.0f, 0.0f, 0.0f, 1.0f), "未启动");
            }

            ImGui::Separator();

            // AI智能脚本开关
            if (ImGui::Checkbox("AI智能脚本", &g_ai_script_enabled)) {
                std::cout << "🤖 AI智能脚本: " << (g_ai_script_enabled ? "已开启" : "已关闭") << std::endl;
            }

            // 显示AI智能脚本状态
            if (g_ai_script_enabled) {
                ImGui::TextColored(ImVec4(0.0f, 1.0f, 0.0f, 1.0f), "AI智能脚本: 已启用");

                // 显示当前区域信息
                std::string currentRegion;
                {
                    std::lock_guard<std::mutex> lock(g_region_mutex);
                    currentRegion = g_current_region;
                }
                ImGui::Text("当前检测区域: %s", currentRegion.c_str());
            }
            else {
                ImGui::TextColored(ImVec4(1.0f, 0.0f, 0.0f, 1.0f), "AI智能脚本: 已禁用");
                ImGui::TextColored(ImVec4(0.7f, 0.7f, 0.7f, 1.0f), "将执行原始脚本逻辑");
            }

            ImGui::Separator();

            // 添加说明信息
            ImGui::TextWrapped("AI智能脚本说明:");
            ImGui::BulletText("开启后根据AI检测的位置智能选择脚本");
            ImGui::BulletText("例: R键绑定多个脚本时，自动选择匹配当前区域的脚本");
            ImGui::BulletText("按 '=' 键快速切换AI智能脚本开关");

            // 显示开关说明
            ImGui::Text("按 \"0\" 键切换工具状态");
            ImGui::Text("按 \"=\" 键切换AI智能脚本");
            ImGui::Text("按 Home 键显示/隐藏主窗口");

            ImGui::Separator();

            // 显示当前时间，使用安全的localtime_s函数
            time_t now = time(0);
            struct tm timeinfo;
            if (localtime_s(&timeinfo, &now) == 0) {
                char timeStr[64];
                strftime(timeStr, sizeof(timeStr), "%Y-%m-%d %H:%M:%S", &timeinfo);
                ImGui::Text("当前时间: %s", timeStr);
            }

            ImGui::Separator();

            // 添加预设选择按钮
            ShowPresetButtons();

            ImGui::Separator();

            // 删除以下触发键设置代码
            // // 触发键设置
            // if (ImGui::Button(("触发键: " + GetKeyName(g_config.trigger_key)).c_str()))
            //     StartListeningForKey(&g_config.trigger_key);

            // // 显示当前状态
            // if (g_is_listening_for_key) {
            //     ImGui::TextColored(ImVec4(1.0f, 0.0f, 0.0f, 1.0f), "请按下新的按键...");
            // }

            ImGui::Separator();

            // 脚本加载部分
            ShowScriptLoadingSection();

            ImGui::Separator();

            // 说明
            ImGui::TextWrapped("评论有礼：");
            ImGui::BulletText("如果觉得工具不错");
            ImGui::BulletText("好评加嗮图");
            ImGui::BulletText("卡密发送给客服 卡密会额外增加一个月时间 ");
            ImGui::EndChild();

            // 右侧面板 - 键位设置和脚本列表
            ImGui::SameLine();
            ImGui::BeginChild("RightPanel", ImVec2(0, 0), true);

            // 键位设置部分
            ImGui::Text("键位设置(设置必须和游戏里一模一样)");

            // 基础操作键位
            if (ImGui::TreeNode("基础操作键位")) {
                // 投篮键
                if (ImGui::Button(("投篮键: " + GetKeyName(g_keys.shoot_key)).c_str())) {
                    ImGui::OpenPopup("设置投篮键");
                }
                ShowKeySelectionPopup("设置投篮键", g_keys.shoot_key);

                // 背身键
                if (ImGui::Button(("背身键: " + GetKeyName(g_keys.post_up_key)).c_str())) {
                    ImGui::OpenPopup("设置背身键");
                }
                ShowKeySelectionPopup("设置背身键", g_keys.post_up_key);

                // 传球键
                if (ImGui::Button(("传球键: " + GetKeyName(g_keys.pass_key)).c_str())) {
                    ImGui::OpenPopup("设置传球键");
                }
                ShowKeySelectionPopup("设置传球键", g_keys.pass_key);

                // 冲刺键
                if (ImGui::Button(("冲刺键: " + GetKeyName(g_keys.sprint_key)).c_str())) {
                    ImGui::OpenPopup("设置冲刺键");
                }
                ShowKeySelectionPopup("设置冲刺键", g_keys.sprint_key);

                // 击地传球键
                if (ImGui::Button(("击地传球键: " + GetKeyName(g_keys.bounce_pass_key)).c_str())) {
                    ImGui::OpenPopup("设置击地传球键");
                }
                ShowKeySelectionPopup("设置击地传球键", g_keys.bounce_pass_key);

                // 高吊传球键
                if (ImGui::Button(("高吊传球键: " + GetKeyName(g_keys.lob_pass_key)).c_str())) {
                    ImGui::OpenPopup("设置高吊传球键");
                }
                ShowKeySelectionPopup("设置高吊传球键", g_keys.lob_pass_key);

                ImGui::TreePop();
            }

            // 移动方向键
            if (ImGui::TreeNode("移动方向键")) {
                // 向上移动
                if (ImGui::Button(("向上移动: " + GetKeyName(g_keys.move_up_key)).c_str())) {
                    ImGui::OpenPopup("设置向上移动键");
                }
                ShowKeySelectionPopup("设置向上移动键", g_keys.move_up_key);

                // 向下移动
                if (ImGui::Button(("向下移动: " + GetKeyName(g_keys.move_down_key)).c_str())) {
                    ImGui::OpenPopup("设置向下移动键");
                }
                ShowKeySelectionPopup("设置向下移动键", g_keys.move_down_key);

                // 向左移动
                if (ImGui::Button(("向左移动: " + GetKeyName(g_keys.move_left_key)).c_str())) {
                    ImGui::OpenPopup("设置向左移动键");
                }
                ShowKeySelectionPopup("设置向左移动键", g_keys.move_left_key);

                // 向右移动
                if (ImGui::Button(("向右移动: " + GetKeyName(g_keys.move_right_key)).c_str())) {
                    ImGui::OpenPopup("设置向右移动键");
                }
                ShowKeySelectionPopup("设置向右移动键", g_keys.move_right_key);

                ImGui::TreePop();
            }

            // 特殊方向键
            if (ImGui::TreeNode("特殊方向键")) {
                // 特殊上键
                if (ImGui::Button(("特殊上键: " + GetKeyName(g_keys.special_up_key)).c_str())) {
                    ImGui::OpenPopup("设置特殊上键");
                }
                ShowKeySelectionPopup("设置特殊上键", g_keys.special_up_key);

                // 特殊下键
                if (ImGui::Button(("特殊下键: " + GetKeyName(g_keys.special_down_key)).c_str())) {
                    ImGui::OpenPopup("设置特殊下键");
                }
                ShowKeySelectionPopup("设置特殊下键", g_keys.special_down_key);

                // 特殊左键
                if (ImGui::Button(("特殊左键: " + GetKeyName(g_keys.special_left_key)).c_str())) {
                    ImGui::OpenPopup("设置特殊左键");
                }
                ShowKeySelectionPopup("设置特殊左键", g_keys.special_left_key);

                // 特殊右键
                if (ImGui::Button(("特殊右键: " + GetKeyName(g_keys.special_right_key)).c_str())) {
                    ImGui::OpenPopup("设置特殊右键");
                }
                ShowKeySelectionPopup("设置特殊右键", g_keys.special_right_key);

                ImGui::TreePop();
            }

            // 高阶操作键位
            if (ImGui::TreeNode("高阶操作键位")) {
                // 特殊斜向操作 (放在前面，因为更基础和常用)
                if (ImGui::CollapsingHeader("特殊斜向操作")) {
                    // 按照左上、右上、左下、右下的顺序排列，更符合直觉
                    // 左上键
                    if (ImGui::Button(("特殊方向左上键: " + GetKeyName(g_keys.special_left_up_key)).c_str())) {
                        ImGui::OpenPopup("设置特殊方向左上键");
                    }
                    ShowKeySelectionPopup("设置特殊方向左上键", g_keys.special_left_up_key);

                    // 右上键
                    if (ImGui::Button(("特殊方向右上键: " + GetKeyName(g_keys.special_right_up_key)).c_str())) {
                        ImGui::OpenPopup("设置特殊方向右上键");
                    }
                    ShowKeySelectionPopup("设置特殊方向右上键", g_keys.special_right_up_key);

                    // 左下键
                    if (ImGui::Button(("特殊方向左下键: " + GetKeyName(g_keys.special_left_down_key)).c_str())) {
                        ImGui::OpenPopup("设置特殊方向左下键");
                    }
                    ShowKeySelectionPopup("设置特殊方向左下键", g_keys.special_left_down_key);

                    // 右下键
                    if (ImGui::Button(("特殊方向右下键: " + GetKeyName(g_keys.special_right_down_key)).c_str())) {
                        ImGui::OpenPopup("设置特殊方向右下键");
                    }
                    ShowKeySelectionPopup("设置特殊方向右下键", g_keys.special_right_down_key);
                }

                // 转身操作 (放在后面，因为是更高级的操作)
                if (ImGui::CollapsingHeader("转身操作")) {
                    // 按照左手、右手的顺序，每种动作放一起
                    // 左手操作
                    if (ImGui::Button(("左手持球半转身键: " + GetKeyName(g_keys.left_hand_spin_key)).c_str())) {
                        ImGui::OpenPopup("设置左手持球半转身键");
                    }
                    ShowKeySelectionPopup("设置左手持球半转身键", g_keys.left_hand_spin_key);

                    if (ImGui::Button(("左手持球转身键: " + GetKeyName(g_keys.left_hand_turn_key)).c_str())) {
                        ImGui::OpenPopup("设置左手持球转身键");
                    }
                    ShowKeySelectionPopup("设置左手持球转身键", g_keys.left_hand_turn_key);

                    // 右手操作
                    if (ImGui::Button(("右手持球半转身键: " + GetKeyName(g_keys.right_hand_spin_key)).c_str())) {
                        ImGui::OpenPopup("设置右手持球半转身键");
                    }
                    ShowKeySelectionPopup("设置右手持球半转身键", g_keys.right_hand_spin_key);

                    if (ImGui::Button(("右手持球转身键: " + GetKeyName(g_keys.right_hand_turn_key)).c_str())) {
                        ImGui::OpenPopup("设置右手持球转身键");
                    }
                    ShowKeySelectionPopup("设置右手持球转身键", g_keys.right_hand_turn_key);
                }

                ImGui::TreePop();
            }

            ImGui::Separator();

            // 脚本列表
            ImGui::Text("脚本列表");

            // 添加清空绑定按钮
            if (ImGui::Button("清空所有脚本绑定")) {
                const auto& scripts = g_script_manager.GetScripts();
                for (size_t i = 0; i < scripts.size(); i++) {
                    Script* script_ptr = g_script_manager.GetScript(i);
                    if (script_ptr) {
                        script_ptr->hotkey = VK_OEM_MINUS;  // 使用'-'键表示未绑定
                    }
                }
                g_script_manager.SaveScriptConfig();
            }
            ImGui::SameLine();
            ImGui::TextDisabled("(?)");
            if (ImGui::IsItemHovered()) {
                ImGui::BeginTooltip();
                ImGui::PushTextWrapPos(ImGui::GetFontSize() * 35.0f);
                ImGui::TextUnformatted("点击此按钮将清空所有脚本的热键绑定");
                ImGui::TextUnformatted("清空后所有脚本热键将显示为\"未设置\"");
                ImGui::PopTextWrapPos();
                ImGui::EndTooltip();
            }

            // 添加标签筛选UI
            ShowScriptTagFilter();
            ImGui::Separator();

            // 获取所有脚本和过滤后的脚本索引
            const auto& scripts = g_script_manager.GetScripts();
            std::vector<size_t> filteredIndices = GetFilteredScriptIndices(scripts);

            const int maxScriptsPerRow = 6; // 每行最多显示6个脚本
            const float scriptItemWidth = ImGui::GetWindowWidth() / maxScriptsPerRow - 10.0f;

            // 根据过滤后的脚本数量动态计算所需行数
            const int filteredScripts = static_cast<int>(filteredIndices.size());
            const int actualRows = (filteredScripts + maxScriptsPerRow - 1) / maxScriptsPerRow; // 向上取整
            const int minRows = 2; // 最少显示2行，确保有足够的显示空间
            const int maxRows = 10; // 最多显示10行，避免窗口过高
            const int displayRows = std::max(minRows, std::min(actualRows, maxRows));
            const float contentHeight = displayRows * 100.0f; // 每行高度约100像素

            // 创建可滚动的子窗口
            ImGui::BeginChild("ScriptList", ImVec2(0, contentHeight), true,
                ImGuiWindowFlags_HorizontalScrollbar | ImGuiWindowFlags_AlwaysHorizontalScrollbar);

            int currentCol = 0; // 当前列计数
            int currentRow = 0; // 当前行计数

            // 显示过滤后的脚本
            for (size_t idx = 0; idx < filteredIndices.size(); idx++) {
                size_t i = filteredIndices[idx];  // 获取原始脚本索引
                const auto& script = scripts[i];

                // 开始新的脚本项，固定宽度确保横向排列整齐
                ImGui::BeginGroup();
                ImGui::PushItemWidth(scriptItemWidth);

                // 显示脚本名称，确保名称不为空
                std::string displayName = script.name.empty() ? "未命名脚本" : script.name;
                ImGui::TextWrapped("%s", displayName.c_str());

                // 显示热键按钮，确保热键有效
                std::string keyName = GetKeyName(script.hotkey);
                std::string buttonLabel = "热键: " + (keyName.empty() ? "未设置" : keyName);

                if (ImGui::Button((buttonLabel + "##" + std::to_string(i)).c_str(), ImVec2(scriptItemWidth - 10.0f, 0))) {
                    m_is_listening_for_script_index = i;
                    ImGui::OpenPopup(("设置脚本热键##" + std::to_string(i)).c_str());
                }

                // 显示脚本描述，只有当描述不为空时才显示
                if (!script.description.empty()) {
                    ImGui::TextColored(ImVec4(0.7f, 0.7f, 0.7f, 1.0f), "%s", script.description.c_str());
                }

                ImGui::PopItemWidth();
                ImGui::EndGroup();

                // 热键设置弹窗 - 保持原有逻辑不变
                if (ImGui::BeginPopupModal(("设置脚本热键##" + std::to_string(i)).c_str(), NULL, ImGuiWindowFlags_AlwaysAutoResize)) {
                    ImGui::Text("请按下要绑定给 \"%s\" 的按键...", displayName.c_str());
                    ImGui::Text("当前选择: %s", keyName.c_str());

                    // 当弹窗打开时，启动按键监听线程
                    if (!m_is_listening_for_key && m_is_listening_for_script_index == i) {
                        Script* script_ptr = g_script_manager.GetScript(i);
                        if (script_ptr) {
                            try {
                                StartKeyListeningThread(&(script_ptr->hotkey));
                            }
                            catch (const std::exception& e) {
                                MessageBoxA(NULL, (std::string("设置热键失败: ") + e.what()).c_str(),
                                    "热键设置错误", MB_OK | MB_ICONERROR);
                                m_is_listening_for_script_index = -1;
                                ImGui::CloseCurrentPopup();
                            }
                        }
                    }

                    if (ImGui::Button("取消")) {
                        StopKeyListeningThread();
                        m_is_listening_for_script_index = -1;
                        ImGui::CloseCurrentPopup();
                    }

                    ImGui::EndPopup();
                }
                else {
                    // 在热键设置弹窗关闭后保存配置
                    if (m_is_listening_for_key && m_is_listening_for_script_index == i) {
                        StopKeyListeningThread();
                        m_is_listening_for_script_index = -1;
                        try {
                            // 保存脚本配置
                            g_script_manager.SaveScriptConfig();
                        }
                        catch (const std::exception& e) {
                            MessageBoxA(NULL, (std::string("保存配置失败: ") + e.what()).c_str(),
                                "配置保存错误", MB_OK | MB_ICONERROR);
                        }
                    }
                }

                // 横向布局：控制每行最多6个脚本
                currentCol++;

                // 如果还有下一个脚本且当前行未满6个，则并排显示
                if (idx + 1 < filteredIndices.size() && currentCol < maxScriptsPerRow) {
                    ImGui::SameLine();
                }
                else {
                    // 否则换行并重置列计数
                    currentCol = 0;
                    currentRow++;
                }
            }

            // 显示脚本统计信息
            const int totalScripts = static_cast<int>(scripts.size());
            if (totalScripts > 0) {
                ImGui::Separator();
                if (filteredScripts < totalScripts) {
                    ImGui::Text("显示 %d 个脚本 (共 %d 个)", filteredScripts, totalScripts);
                }
                else {
                    ImGui::Text("共加载 %d 个脚本", totalScripts);
                }

                if (actualRows > maxRows) {
                    ImGui::TextColored(ImVec4(1.0f, 0.5f, 0.0f, 1.0f),
                        "注意：脚本较多，请使用滚动条查看所有脚本");
                }
            }

            ImGui::EndChild();

            ImGui::EndChild();

            ImGui::End();
        }

        // 渲染
        ImGui::Render();
        const float clear_color_with_alpha[4] = { 0.45f, 0.55f, 0.60f, 1.00f };
        m_pd3dDeviceContext->OMSetRenderTargets(1, &m_mainRenderTargetView, nullptr);
        m_pd3dDeviceContext->ClearRenderTargetView(m_mainRenderTargetView, clear_color_with_alpha);
        ImGui_ImplDX11_RenderDrawData(ImGui::GetDrawData());
        m_pSwapChain->Present(1, 0);
    }

    // DirectX相关函数
    bool CreateDeviceD3D(HWND hWnd) {
        DXGI_SWAP_CHAIN_DESC sd;
        ZeroMemory(&sd, sizeof(sd));
        sd.BufferCount = 2;
        sd.BufferDesc.Width = 0;
        sd.BufferDesc.Height = 0;
        sd.BufferDesc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
        sd.BufferDesc.RefreshRate.Numerator = 60;
        sd.BufferDesc.RefreshRate.Denominator = 1;
        sd.Flags = DXGI_SWAP_CHAIN_FLAG_ALLOW_MODE_SWITCH;
        sd.BufferUsage = DXGI_USAGE_RENDER_TARGET_OUTPUT;
        sd.OutputWindow = hWnd;
        sd.SampleDesc.Count = 1;
        sd.SampleDesc.Quality = 0;
        sd.Windowed = TRUE;
        sd.SwapEffect = DXGI_SWAP_EFFECT_DISCARD;

        UINT createDeviceFlags = 0;
        D3D_FEATURE_LEVEL featureLevel;
        const D3D_FEATURE_LEVEL featureLevelArray[2] = { D3D_FEATURE_LEVEL_11_0, D3D_FEATURE_LEVEL_10_0, };

        if (D3D11CreateDeviceAndSwapChain(nullptr, D3D_DRIVER_TYPE_HARDWARE, nullptr, createDeviceFlags,
            featureLevelArray, 2, D3D11_SDK_VERSION, &sd, &m_pSwapChain, &m_pd3dDevice,
            &featureLevel, &m_pd3dDeviceContext) != S_OK)
            return false;

        CreateRenderTarget();
        return true;
    }

    void CleanupDeviceD3D() {
        CleanupRenderTarget();
        if (m_pSwapChain) { m_pSwapChain->Release(); m_pSwapChain = nullptr; }
        if (m_pd3dDeviceContext) { m_pd3dDeviceContext->Release(); m_pd3dDeviceContext = nullptr; }
        if (m_pd3dDevice) { m_pd3dDevice->Release(); m_pd3dDevice = nullptr; }
    }

    void CreateRenderTarget() {
        ID3D11Texture2D* pBackBuffer;
        m_pSwapChain->GetBuffer(0, IID_PPV_ARGS(&pBackBuffer));
        m_pd3dDevice->CreateRenderTargetView(pBackBuffer, nullptr, &m_mainRenderTargetView);
        pBackBuffer->Release();
    }

    void CleanupRenderTarget() {
        if (m_mainRenderTargetView) { m_mainRenderTargetView->Release(); m_mainRenderTargetView = nullptr; }
    }

    // 添加按键设置相关的方法
    void ShowKeySelectionPopup(const char* popup_name, int& key_code) {
        if (ImGui::BeginPopupModal(popup_name, NULL, ImGuiWindowFlags_AlwaysAutoResize)) {
            ImGui::Text("请按下要使用的按键...");
            ImGui::Text("当前选择: %s", GetKeyName(key_code).c_str());

            // 当弹窗打开时，启动按键监听线程
            if (!m_is_listening_for_key) {
                StartKeyListeningThread(&key_code);
            }

            if (ImGui::Button("取消")) {
                StopKeyListeningThread();
                ImGui::CloseCurrentPopup();
            }

            ImGui::EndPopup();
        }
        else {
            // 如果弹窗关闭，停止监听线程
            if (m_is_listening_for_key && m_target_key_code == &key_code) {
                StopKeyListeningThread();
            }
        }
    }

    // 在MainWindow类中添加以下函数来显示脚本加载状态
    void ShowScriptLoadingSection() {
        static bool loadingSuccess = false;
        static std::string loadingMessage;

        if (ImGui::Button("刷新脚本")) {
            try {
                // 先保存当前脚本配置
                g_script_manager.SaveScriptConfig();

                // 重新加载脚本目录中的所有脚本
                ScriptRegistrar::ClearScripts();
                ScriptRegistrar::LoadScriptsFromDirectory("scripts");

                // 为没有热键的脚本设置默认F10热键
                const auto& scripts = g_script_manager.GetScripts();
                bool hasChanges = false;
                for (size_t i = 0; i < scripts.size(); i++) {
                    Script* script_ptr = g_script_manager.GetScript(i);
                    if (script_ptr && script_ptr->hotkey == 0) {
                        script_ptr->hotkey = VK_F10;
                        hasChanges = true;
                    }
                }

                // 如果有修改，保存配置
                if (hasChanges) {
                    g_script_manager.SaveScriptConfig();
                }

                loadingSuccess = true;
                loadingMessage = "脚本加载成功！";
            }
            catch (const std::exception& e) {
                loadingSuccess = false;
                loadingMessage = std::string("脚本加载失败: ") + e.what();
            }
        }

        ImGui::SameLine();
        ImGui::TextDisabled("(?)");
        if (ImGui::IsItemHovered()) {
            ImGui::BeginTooltip();
            ImGui::PushTextWrapPos(ImGui::GetFontSize() * 35.0f);
            ImGui::TextUnformatted("将脚本文件(.txt)放入scripts文件夹中，然后点击此按钮加载");
            ImGui::TextUnformatted("请确保脚本文件使用UTF-8编码格式保存");
            ImGui::PopTextWrapPos();
            ImGui::EndTooltip();
        }

        // 显示加载状态
        if (!loadingMessage.empty()) {
            ImGui::SameLine();
            if (loadingSuccess) {
                ImGui::TextColored(ImVec4(0.0f, 1.0f, 0.0f, 1.0f), "%s", loadingMessage.c_str());
            }
            else {
                ImGui::TextColored(ImVec4(1.0f, 0.0f, 0.0f, 1.0f), "%s", loadingMessage.c_str());
            }
        }
    }

    ~MainWindow() {
        StopKeyListeningThread();
        ImGui_ImplDX11_Shutdown();
        ImGui_ImplWin32_Shutdown();
        ImGui::DestroyContext();
        CleanupDeviceD3D();
    }
};

int WINAPI wWinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPWSTR lpCmdLine, int nShowCmd) {
    // 分配控制台窗口以显示输出
    AllocConsole();

    // 设置控制台编码为UTF-8，正确支持中文
    SetConsoleCP(CP_UTF8);          // 设置输入代码页为UTF-8
    SetConsoleOutputCP(CP_UTF8);    // 设置输出代码页为UTF-8

    // 重新定向标准输入输出
    freopen_s((FILE**)stdout, "CONOUT$", "w", stdout);
    freopen_s((FILE**)stderr, "CONOUT$", "w", stderr);
    freopen_s((FILE**)stdin, "CONIN$", "r", stdin);

    // 设置locale以支持UTF-8
    setlocale(LC_ALL, ".UTF8");

    // 设置控制台标题（使用UTF-8编码）
    SetConsoleTitle(TEXT("篮球AI检测客户端 - 区域数据监听器"));

    std::cout << "========================================" << std::endl;
    std::cout << "🏀 篮球AI检测客户端启动中..." << std::endl;
    std::cout << "========================================" << std::endl;

    // 创建并启动Socket客户端
    g_socket_client = std::make_unique<RegionSocketClient>("127.0.0.1", 12345);
    if (!g_socket_client->Start()) {
        std::cerr << "❌ Socket客户端启动失败！" << std::endl;
        return -1;
    }

    // 创建scripts文件夹（如果不存在）
    CreateDirectoryA("scripts", NULL);

    // 初始化验证器
    if (!LicenseVerifier::Initialize()) {
        MessageBoxA(nullptr, "初始化验证器失败！", "错误", MB_OK | MB_ICONERROR);
        return -1;
    }

    MainWindow mainWindow;
    if (!mainWindow.Initialize(hInstance)) {
        return 1;
    }

    // 启动监控线程
    std::thread monitor(MonitorThread);
    monitor.detach();

    std::cout << "✅ 篮球AI检测客户端完全启动!" << std::endl;
    std::cout << "正在监听来自AI检测服务器的区域数据..." << std::endl;
    std::cout << "========================================" << std::endl;

    // 消息循环
    MSG msg;
    ZeroMemory(&msg, sizeof(msg));
    while (msg.message != WM_QUIT) {
        if (PeekMessage(&msg, nullptr, 0U, 0U, PM_REMOVE)) {
            TranslateMessage(&msg);
            DispatchMessage(&msg);
            continue;
        }
        mainWindow.RenderFrame();
    }

    // 清理资源
    if (g_socket_client) {
        g_socket_client->Stop();
        g_socket_client.reset();
    }

    // 关闭控制台
    FreeConsole();

    return 0;
}

int main() {
    return wWinMain(GetModuleHandle(NULL), NULL, GetCommandLine(), SW_SHOWDEFAULT);
}